#
# Copyright (C) 2020 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
def test(name,
         input0,
         output_width,
         output_height,
         layout,
         align_corners,
         half_pixel_centers,
         output0,
         input0_data,
         output_data,
         add_quant_variations=True):
  model = Model().Operation("RESIZE_BILINEAR", input0, output_width,
                            output_height, layout, align_corners,
                            half_pixel_centers).To(output0)
  quant8 = DataTypeConverter().Identify({
      input0: ["TENSOR_QUANT8_ASYMM", 0.5, 128],
      output0: ["TENSOR_QUANT8_ASYMM", 0.5, 128],
  })
  quant8_signed = DataTypeConverter().Identify({
      input0: ["TENSOR_QUANT8_ASYMM_SIGNED", 0.5, 0],
      output0: ["TENSOR_QUANT8_ASYMM_SIGNED", 0.5, 0],
  })
  variations = ["float16", "relaxed"]
  if add_quant_variations:
    variations.extend([quant8, quant8_signed])

  example = Example({
      input0: input0_data,
      output0: output_data,
  },
                    model=model,
                    name=name).AddVariations(*variations)


test(
    name="half_pixel_centers_2x2x2x1_to_2x3x3x1",
    input0=Input("input0", "TENSOR_FLOAT32", "{2, 2, 2, 1}"),
    output_width=Int32Scalar("output_width", 3),
    output_height=Int32Scalar("output_height", 3),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", False),
    half_pixel_centers=BoolScalar("half_pixel_centers", True),
    output0=Output("output0", "TENSOR_FLOAT32", "{2, 3, 3, 1}"),
    input0_data=[1, 2, 3, 4, 1, 2, 3, 4],
    output_data=[
        1, 1.5, 2, 2, 2.5, 3, 3, 3.5, 4, 1, 1.5, 2, 2, 2.5, 3, 3, 3.5, 4
    ],
)

test(
    name="half_pixel_centers_2x14x13x3_to_2x6x7x3",
    input0=Input("input0", "TENSOR_FLOAT32", "{2, 14, 13, 3}"),
    output_width=Int32Scalar("output_width", 7),
    output_height=Int32Scalar("output_height", 6),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", False),
    half_pixel_centers=BoolScalar("half_pixel_centers", True),
    output0=Output("output0", "TENSOR_FLOAT32", "{2, 6, 7, 3}"),
    input0_data=[
        0.827637, 0.98301, 0.250087, 0.575817, 0.063061, 0.534553, 0.675679,
        0.694844, 0.497918, 0.664793, 0.0200533, 0.577735, 0.814857, 0.843088,
        0.328894, 0.700158, 0.540654, 0.483906, 0.0713209, 0.231078, 0.0430164,
        0.146823, 0.556193, 0.372893, 0.42612, 0.640223, 0.326292, 0.489664,
        0.267468, 0.413154, 0.88774, 0.610036, 0.942792, 0.23379, 0.0979913,
        0.458068, 0.223809, 0.0780525, 0.770556, 0.391897, 0.891485, 0.364972,
        0.847238, 0.428266, 0.660148, 0.990963, 0.670023, 0.982757, 0.0835382,
        0.208294, 0.689837, 0.673272, 0.317975, 0.382154, 0.368691, 0.408292,
        0.0825884, 0.979362, 0.667223, 0.452756, 0.531345, 0.361022, 0.974065,
        0.203924, 0.0682611, 0.930153, 0.272573, 0.398286, 0.582229, 0.552098,
        0.236405, 0.831928, 0.253404, 0.102948, 0.701941, 0.472118, 0.415567,
        0.830793, 0.995918, 0.873392, 0.148214, 0.385363, 0.900278, 0.0703746,
        0.108, 0.528804, 0.944798, 0.949568, 0.122064, 0.840799, 0.532888,
        0.0518012, 0.966821, 0.611478, 0.0889368, 0.591055, 0.71221, 0.399697,
        0.736551, 0.675313, 0.0995619, 0.975917, 0.329392, 0.513981, 0.563206,
        0.86733, 0.716193, 0.2221, 0.215225, 0.226138, 0.661863, 0.465466,
        0.164099, 0.807117, 0.22327, 0.0895369, 0.982572, 0.429804, 0.0558029,
        0.799344, 0.169512, 0.569326, 0.135653, 0.777692, 0.11906, 0.362015,
        0.40525, 0.0627866, 0.515949, 0.000611305, 0.583503, 0.947306, 0.869187,
        0.869985, 0.945147, 0.570262, 0.709512, 0.0355313, 0.446349, 0.80268,
        0.766533, 0.161885, 0.0908636, 0.450652, 0.111231, 0.346606, 0.84161,
        0.524124, 0.607721, 0.393173, 0.103109, 0.943106, 0.453668, 0.598608,
        0.323443, 0.79512, 0.227289, 0.13272, 0.944727, 0.653672, 0.688597,
        0.40432, 0.0568643, 0.568614, 0.962205, 0.94967, 0.0944915, 0.10954,
        0.333137, 0.815286, 0.0110344, 0.3073, 0.661422, 0.884207, 0.744278,
        0.89397, 0.00762653, 0.703588, 0.812244, 0.194066, 0.174366, 0.884352,
        0.898688, 0.114173, 0.479097, 0.452601, 0.00374603, 0.791901, 0.0691,
        0.688598, 0.329226, 0.872065, 0.056097, 0.810847, 0.0154784, 0.986826,
        0.133701, 0.84835, 0.301012, 0.429658, 0.434824, 0.63403, 0.109551,
        0.594964, 0.414044, 0.512716, 0.986874, 0.365579, 0.129968, 0.553444,
        0.518211, 0.111823, 0.290679, 0.335546, 0.0241963, 0.420873, 0.831382,
        0.0305338, 0.779728, 0.351471, 0.606134, 0.0897753, 0.118784, 0.761163,
        0.641289, 0.883826, 0.877834, 0.854164, 0.725214, 0.69578, 0.913621,
        0.912614, 0.81458, 0.547955, 0.0563488, 0.976676, 0.0151438, 0.767199,
        0.173275, 0.543553, 0.0929028, 0.471326, 0.181049, 0.0777475, 0.278217,
        0.101287, 0.421686, 0.55636, 0.344953, 0.681681, 0.01902, 0.646946,
        0.310552, 0.622743, 0.709501, 0.0542059, 0.559718, 0.0417655, 0.847932,
        0.35674, 0.280232, 0.605592, 0.267366, 0.180051, 0.522576, 0.039578,
        0.717534, 0.109444, 0.384435, 0.365782, 0.800604, 0.000903726, 0.11138,
        0.615857, 0.335662, 0.824687, 0.909823, 0.479827, 0.509285, 0.748855,
        0.546289, 0.240599, 0.135308, 0.487517, 0.543101, 0.918314, 0.736137,
        0.167971, 0.907674, 0.86526, 0.92979, 0.347061, 0.0289562, 0.183843,
        0.109654, 0.25219, 0.913093, 0.211304, 0.0239445, 0.321891, 0.0132581,
        0.491192, 0.0598186, 0.638829, 0.922965, 0.970296, 0.688952, 0.129948,
        0.147574, 0.260364, 0.556817, 0.676236, 0.967277, 0.741944, 0.455842,
        0.372167, 0.594965, 0.194687, 0.110991, 0.828473, 0.81903, 0.300324,
        0.799584, 0.888032, 0.401994, 0.751757, 0.341877, 0.168591, 0.958628,
        0.989044, 0.368874, 0.499499, 0.984334, 0.405659, 0.259569, 0.151238,
        0.764487, 0.784618, 0.441848, 0.972594, 0.131611, 0.0371498, 0.0693474,
        0.416801, 0.324805, 0.0487738, 0.336595, 0.00118566, 0.484068, 0.856299,
        0.780161, 0.168135, 0.902528, 0.00840139, 0.0114626, 0.363331, 0.368221,
        0.560426, 0.19353, 0.724254, 0.954187, 0.68854, 0.160637, 0.777354,
        0.692321, 0.157433, 0.369148, 0.192732, 0.949813, 0.307579, 0.958941,
        0.483992, 0.602389, 0.401443, 0.250001, 0.000342846, 0.120461, 0.875903,
        0.407285, 0.973199, 0.0558323, 0.293595, 0.362371, 0.619545, 0.366578,
        0.970044, 0.883848, 0.314051, 0.272863, 0.910542, 0.335767, 0.364503,
        0.217633, 0.586304, 0.62947, 0.93676, 0.747138, 0.81246, 0.643208,
        0.740491, 0.55069, 0.336218, 0.980699, 0.261752, 0.0782728, 0.45507,
        0.485201, 0.443404, 0.80983, 0.146657, 0.313562, 0.137156, 0.634548,
        0.418098, 0.592411, 0.0155032, 0.336527, 0.273182, 0.306312, 0.4013,
        0.519121, 0.403324, 0.0888798, 0.553309, 0.637661, 0.643027, 0.711509,
        0.169574, 0.531634, 0.184579, 0.802464, 0.262788, 0.582185, 0.838128,
        0.903281, 0.89136, 0.400026, 0.962787, 0.393862, 0.0912223, 0.924886,
        0.732427, 0.922716, 0.320009, 0.017068, 0.246307, 0.0766512, 0.28073,
        0.956669, 0.000580311, 0.0147009, 0.468664, 0.606133, 0.589364,
        0.960154, 0.457708, 0.0992912, 0.229694, 0.722875, 0.0905478, 0.492376,
        0.460984, 0.822356, 0.631099, 0.120013, 0.9392, 0.025933, 0.493204,
        0.0777059, 0.996434, 0.90678, 0.570172, 0.137066, 0.587406, 0.152012,
        0.330419, 0.597728, 0.653196, 0.210094, 0.0150489, 0.499774, 0.512619,
        0.662554, 0.0727493, 0.902094, 0.35155, 0.660929, 0.538246, 0.493768,
        0.436716, 0.347757, 0.718451, 0.0751067, 0.591153, 0.476367, 0.871649,
        0.770177, 0.49412, 0.207645, 0.300805, 0.944687, 0.540949, 0.334802,
        0.658609, 0.280322, 0.0614085, 0.523165, 0.480998, 0.452187, 0.396826,
        0.814263, 0.444388, 0.0979315, 0.613517, 0.591663, 0.301699, 0.0100771,
        0.646675, 0.775883, 0.0320501, 0.362856, 0.0123467, 0.335263, 0.785518,
        0.0548519, 0.236233, 0.982908, 0.397743, 0.196763, 0.273222, 0.738824,
        0.921409, 0.746146, 0.184881, 0.780506, 0.790668, 0.838174, 0.140729,
        0.344877, 0.437886, 0.270443, 0.113299, 0.0567528, 0.872003, 0.461713,
        0.396225, 0.334525, 0.571957, 0.406145, 0.655128, 0.111169, 0.512328,
        0.579207, 0.681437, 0.313002, 0.526335, 0.682192, 0.543195, 0.883647,
        0.680931, 0.936027, 0.887212, 0.254619, 0.537763, 0.929323, 0.899553,
        0.583033, 0.479059, 0.58146, 0.605686, 0.0800954, 0.554806, 0.805312,
        0.730701, 0.461951, 0.861251, 0.338622, 0.949928, 0.676404, 0.779811,
        0.66965, 0.413489, 0.244133, 0.253003, 0.668361, 0.861042, 0.111543,
        0.374333, 0.620414, 0.730657, 0.257099, 0.992287, 0.962735, 0.48282,
        0.103903, 0.120562, 0.371291, 0.828842, 0.19805, 0.3552, 0.139294,
        0.983518, 0.773706, 0.304598, 0.321077, 0.148372, 0.975174, 0.902274,
        0.021295, 0.109916, 0.0552523, 0.129696, 0.834911, 0.35488, 0.332393,
        0.638469, 0.505043, 0.607849, 0.444755, 0.507821, 0.0467358, 0.966812,
        0.403, 0.907803, 0.491011, 0.330456, 0.400001, 0.246176, 0.0303432,
        0.744088, 0.712685, 0.382909, 0.0748539, 0.0552307, 0.250542, 0.432398,
        0.269405, 0.965367, 0.255643, 0.00846922, 0.477259, 0.956944, 0.263619,
        0.472611, 0.818297, 0.637181, 0.837281, 0.29029, 0.52285, 0.411453,
        0.327927, 0.493403, 0.292899, 0.326031, 0.0675526, 0.043205, 0.117051,
        0.782362, 0.974107, 0.713485, 0.192605, 0.757342, 0.0791711, 0.773982,
        0.918075, 0.0538017, 0.0902326, 0.385477, 0.61089, 0.893094, 0.278611,
        0.749566, 0.297577, 0.343, 0.700941, 0.021899, 0.785716, 0.575491,
        0.571529, 0.895896, 0.540937, 0.686329, 0.519124, 0.214197, 0.257485,
        0.479406, 0.723769, 0.133132, 0.654102, 0.464483, 0.520407, 0.739025,
        0.425871, 0.0430206, 0.367081, 0.819075, 0.502319, 0.808982, 0.586205,
        0.693161, 0.578652, 0.670991, 0.564139, 0.168943, 0.578697, 0.0289714,
        0.331961, 0.0185615, 0.981341, 0.79096, 0.683865, 0.203928, 0.860001,
        0.630191, 0.382722, 0.887468, 0.209518, 0.582372, 0.196313, 0.0263103,
        0.305729, 0.76197, 0.52452, 0.178965, 0.719016, 0.605278, 0.117012,
        0.838881, 0.10849, 0.0151021, 0.695286, 0.134018, 0.923889, 0.361624,
        0.801152, 0.345261, 0.992369, 0.279426, 0.574633, 0.724684, 0.264346,
        0.216093, 0.622767, 0.908065, 0.882313, 0.796358, 0.154155, 0.308829,
        0.7388, 0.483272, 0.726135, 0.0916826, 0.831043, 0.147788, 0.982867,
        0.639764, 0.308342, 0.480933, 0.682025, 0.0305799, 0.448529, 0.485899,
        0.684693, 0.604456, 0.221594, 0.461711, 0.350639, 0.58591, 0.959523,
        0.831963, 0.914305, 0.361302, 0.620165, 0.268338, 0.296164, 0.0403097,
        0.822701, 0.809845, 0.524713, 0.645126, 0.207234, 0.188755, 0.135637,
        0.756508, 0.171036, 0.332887, 0.513941, 0.154996, 0.555009, 0.98974,
        0.303486, 0.473306, 0.430805, 0.187765, 0.377134, 0.612192, 0.902819,
        0.29555, 0.409627, 0.869718, 0.226232, 0.756249, 0.935133, 0.546867,
        0.384815, 0.188028, 0.750887, 0.310038, 0.44034, 0.824127, 0.747097,
        0.128963, 0.24836, 0.528062, 0.553374, 0.689335, 0.126468, 0.175476,
        0.975612, 0.586703, 0.671013, 0.962735, 0.808907, 0.0861794, 0.533115,
        0.101796, 0.0706959, 0.0917822, 0.0471025, 0.777679, 0.952353, 0.385382,
        0.820037, 0.815421, 0.47614, 0.424147, 0.946672, 0.910313, 0.884953,
        0.476174, 0.426554, 0.985588, 0.032336, 0.383584, 0.0543795, 0.745913,
        0.93714, 0.984151, 0.643788, 0.680333, 0.660295, 0.451879, 0.81687,
        0.423028, 0.0934314, 0.317183, 0.32431, 0.798049, 0.301885, 0.482865,
        0.0412549, 0.305506, 0.498927, 0.183137, 0.440198, 0.708901, 0.142019,
        0.74945, 0.744077, 0.406925, 0.502505, 0.977357, 0.981186, 0.454713,
        0.694598, 0.583974, 0.0061208, 0.313697, 0.384455, 0.141156, 0.121607,
        0.225996, 0.252465, 0.0127006, 0.692692, 0.542434, 0.385375, 0.0158385,
        0.469898, 0.88728, 0.580118, 0.516798, 0.124881, 0.17155, 0.10754,
        0.64776, 0.863566, 0.585709, 0.737403, 0.881578, 0.768254, 0.926182,
        0.734802, 0.0924621, 0.560301, 0.973171, 0.197622, 0.39735, 0.831218,
        0.831679, 0.626448, 0.255363, 0.221987, 0.510817, 0.119718, 0.457004,
        0.751342, 0.682048, 0.925028, 0.472245, 0.946413, 0.702793, 0.849434,
        0.835024, 0.757633, 0.227651, 0.15449, 0.581614, 0.344011, 0.27533,
        0.861278, 0.0480329, 0.296223, 0.639267, 0.422976, 0.0219772, 0.547381,
        0.143436, 0.483599, 0.922199, 0.657291, 0.968148, 0.826014, 0.0362797,
        0.285515, 0.754099, 0.819942, 0.07358, 0.485557, 0.0095253, 0.381697,
        0.690303, 0.166019, 0.162123, 0.746008, 0.317472, 0.684392, 0.824646,
        0.189942, 0.155815, 0.865175, 0.508899, 0.498846, 0.510297, 0.314311,
        0.0580601, 0.25675, 0.849481, 0.997329, 0.867238, 0.255343, 0.575945,
        0.929926, 0.164465, 0.382339, 0.0175594, 0.531675, 0.133809, 0.424927,
        0.932461, 0.47343, 0.670699, 0.451685, 0.727976, 0.157594, 0.668399,
        0.431421, 0.692803, 0.309855, 0.320758, 0.600577, 0.851317, 0.110719,
        0.72173, 0.052147, 0.508466, 0.331822, 0.732259, 0.300817, 0.709938,
        0.341123, 0.698617, 0.212518, 0.202616, 0.018757, 0.177552, 0.613048,
        0.758948, 0.547059, 0.400971, 0.115883, 0.256621, 0.727341, 0.47802,
        0.377706, 0.344168, 0.903898, 0.485407, 0.654939, 0.98362, 0.858975,
        0.575978, 0.698721, 0.721705, 0.841963, 0.209065, 0.43947, 0.626849,
        0.616108, 0.503407, 0.551573, 0.393978, 0.0142356, 0.1245, 0.0788039,
        0.329059, 0.540094, 0.963904, 0.579622, 0.806494, 0.378628, 0.793516,
        0.649193, 0.151952, 0.572264, 0.460979, 0.198079, 0.932621, 0.719997,
        0.919287, 0.0693846, 0.840908, 0.843045, 0.055732, 0.457907, 0.992388,
        0.945144, 0.943088, 0.676341, 0.553268, 0.586761, 0.224883, 0.861508,
        0.566519, 0.911452, 0.591536, 0.0733222, 0.170432, 0.351752, 0.157091,
        0.327673, 0.868404, 0.808417, 0.569937, 0.453585, 0.498946, 0.80573,
        0.626621, 0.349575, 0.642979, 0.53113, 0.138728, 0.4121, 0.425972,
        0.807962, 0.831206, 0.839713, 0.129973, 0.553252, 0.147851, 0.733317,
        0.196179
    ],
    output_data=[
        0.631267, 0.658224, 0.451651, 0.711989, 0.526083, 0.772951, 0.685998,
        0.487213, 0.343252, 0.676681, 0.521841, 0.316176, 0.295872, 0.282796,
        0.735267, 0.572824, 0.359158, 0.770879, 0.328285, 0.216591, 0.729258,
        0.318251, 0.275884, 0.514905, 0.67128, 0.102982, 0.405996, 0.124694,
        0.624429, 0.946997, 0.570262, 0.709512, 0.0355313, 0.202523, 0.192552,
        0.495778, 0.406154, 0.533116, 0.521298, 0.386251, 0.589013, 0.648783,
        0.58227, 0.38448, 0.527951, 0.410309, 0.288544, 0.524046, 0.411198,
        0.244783, 0.473302, 0.327214, 0.223712, 0.517082, 0.524867, 0.477849,
        0.207615, 0.625594, 0.597887, 0.716884, 0.722997, 0.564202, 0.543249,
        0.722101, 0.585117, 0.459117, 0.301243, 0.723311, 0.668172, 0.551366,
        0.772401, 0.38182, 0.361603, 0.342651, 0.717503, 0.541077, 0.491924,
        0.402338, 0.371644, 0.490079, 0.398466, 0.346357, 0.547003, 0.447432,
        0.44314, 0.47806, 0.52583, 0.745239, 0.737669, 0.555497, 0.905609,
        0.293674, 0.130434, 0.80983, 0.146657, 0.313562, 0.527374, 0.103938,
        0.34818, 0.448853, 0.375606, 0.178143, 0.643709, 0.370183, 0.579374,
        0.40045, 0.568371, 0.602847, 0.34384, 0.39982, 0.433225, 0.317636,
        0.519918, 0.418223, 0.519364, 0.686467, 0.46176, 0.357139, 0.597116,
        0.428639, 0.372748, 0.447356, 0.728054, 0.466077, 0.223182, 0.471054,
        0.662586, 0.376077, 0.36661, 0.680342, 0.708627, 0.426894, 0.781609,
        0.413516, 0.569702, 0.51284, 0.513902, 0.293259, 0.20138, 0.303709,
        0.335072, 0.613278, 0.578262, 0.595837, 0.653285, 0.442471, 0.545559,
        0.480946, 0.689819, 0.292554, 0.722947, 0.297008, 0.735673, 0.100418,
        0.801456, 0.570554, 0.686329, 0.519124, 0.214197, 0.150897, 0.629145,
        0.501524, 0.179417, 0.47335, 0.706731, 0.611372, 0.677365, 0.634654,
        0.481956, 0.525163, 0.463365, 0.502, 0.43508, 0.565645, 0.266055,
        0.70408, 0.55236, 0.156929, 0.588925, 0.399344, 0.475003, 0.356185,
        0.59649, 0.562223, 0.526428, 0.439555, 0.492657, 0.323885, 0.765127,
        0.429603, 0.466623, 0.327579, 0.381093, 0.570198, 0.257691, 0.463234,
        0.879136, 0.894351, 0.307519, 0.504116, 0.415028, 0.287617, 0.0813091,
        0.632005, 0.578317, 0.746861, 0.448167, 0.365736, 0.404652, 0.543348,
        0.708017, 0.363659, 0.691263, 0.805467, 0.260183, 0.224962, 0.514807,
        0.0318816, 0.350329, 0.746008, 0.317472, 0.684392, 0.859385, 0.463334,
        0.449842, 0.329192, 0.696575, 0.728967, 0.90306, 0.203413, 0.465313,
        0.428212, 0.540184, 0.911023, 0.760282, 0.628308, 0.512895, 0.484829,
        0.43916, 0.453882, 0.298958, 0.352472, 0.242515, 0.563185, 0.635929,
        0.530235, 0.575659, 0.548754, 0.36042, 0.441017, 0.684306, 0.700217
    ],
    add_quant_variations=False,
)

test(
    name="half_pixel_centers_2x6x7x3_to_2x14x13x3",
    input0=Input("input0", "TENSOR_FLOAT32", "{2, 6, 7, 3}"),
    output_width=Int32Scalar("output_width", 13),
    output_height=Int32Scalar("output_height", 14),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", False),
    half_pixel_centers=BoolScalar("half_pixel_centers", True),
    output0=Output("output0", "TENSOR_FLOAT32", "{2, 14, 13, 3}"),
    input0_data=[
        0.597099, 0.717064, 0.696989, 0.37401, 0.587701, 0.52613, 0.45983,
        0.834707, 0.825826, 0.188192, 0.0581832, 0.489487, 0.897472, 0.912709,
        0.338215, 0.114233, 0.975378, 0.189915, 0.885871, 0.0123631, 0.682109,
        0.0637571, 0.021166, 0.990439, 0.405466, 0.245107, 0.235285, 0.670532,
        0.721946, 0.561132, 0.00936949, 0.0526415, 0.447197, 0.755065, 0.597531,
        0.195151, 0.269797, 0.551208, 0.954087, 0.223795, 0.397895, 0.924789,
        0.635592, 0.297786, 0.0237268, 0.101017, 0.491771, 0.715934, 0.546113,
        0.782431, 0.822342, 0.454632, 0.584726, 0.427173, 0.130061, 0.63263,
        0.0130227, 0.469195, 0.689757, 0.02537, 0.630514, 0.985893, 0.257825,
        0.0865368, 0.465077, 0.950239, 0.221726, 0.349166, 0.799219, 0.181399,
        0.312473, 0.0402693, 0.533386, 0.581182, 0.600382, 0.186105, 0.322734,
        0.460414, 0.37009, 0.452929, 0.387669, 0.883785, 0.291214, 0.403792,
        0.288797, 0.114932, 0.441901, 0.369459, 0.00886202, 0.577986, 0.457492,
        0.488561, 0.845964, 0.0515606, 0.210285, 0.14478, 0.935541, 0.874223,
        0.0491093, 0.81723, 0.35612, 0.0562574, 0.763055, 0.364821, 0.731659,
        0.745522, 0.64809, 0.510889, 0.430949, 0.161836, 0.735475, 0.264182,
        0.476501, 0.465461, 0.258138, 0.730769, 0.899318, 0.174359, 0.86886,
        0.756424, 0.685789, 0.229707, 0.443774, 0.275146, 0.658326, 0.0486892,
        0.0457834, 0.787678, 0.663175, 0.481761, 0.573898, 0.439558, 0.56098,
        0.98622, 0.807752, 0.34172, 0.972262, 0.284004, 0.396007, 0.440433,
        0.20663, 0.121019, 0.0240191, 0.0363696, 0.93515, 0.283339, 0.90816,
        0.694879, 0.28891, 0.7681, 0.52112, 0.473317, 0.933415, 0.149368,
        0.179469, 0.922061, 0.625879, 0.916547, 0.112067, 0.974693, 0.553303,
        0.44083, 0.772933, 0.423641, 0.0963937, 0.937332, 0.301141, 0.164467,
        0.977361, 0.224354, 0.84855, 0.728315, 0.564487, 0.733492, 0.911596,
        0.794423, 0.175493, 0.156265, 0.511936, 0.574377, 0.394498, 0.522894,
        0.0984874, 0.921798, 0.850203, 0.596674, 0.0368204, 0.326229, 0.281598,
        0.939939, 0.487391, 0.596167, 0.549989, 0.0975398, 0.983239, 0.993055,
        0.0635948, 0.749972, 0.177322, 0.771397, 0.777728, 0.531868, 0.31498,
        0.334786, 0.633789, 0.959249, 0.573456, 0.429871, 0.238746, 0.376446,
        0.0555074, 0.506892, 0.565241, 0.897762, 0.409047, 0.890007, 0.394771,
        0.923041, 0.0133429, 0.940546, 0.601826, 0.117889, 0.341266, 0.77822,
        0.948948, 0.675557, 0.620504, 0.274183, 0.697336, 0.170604, 0.726621,
        0.583086, 0.0914261, 0.621153, 0.158024, 0.481524, 0.801537, 0.682243,
        0.469472, 0.0622474, 0.0587367, 0.986715, 0.24073, 0.454288, 0.199331,
        0.830628, 0.866293, 0.504453, 0.62714, 0.760198, 0.848832, 0.467382
    ],
    output_data=[
        0.597099, 0.717064, 0.696989, 0.528456, 0.67726, 0.644417, 0.408331,
        0.607603, 0.552416, 0.407018, 0.682704, 0.641398, 0.453228, 0.815707,
        0.802773, 0.334458, 0.476311, 0.670593, 0.188192, 0.0581834, 0.489487,
        0.570112, 0.518313, 0.408033, 0.837223, 0.91753, 0.326807, 0.415478,
        0.951275, 0.246953, 0.232946, 0.827222, 0.265637, 0.648444, 0.308675,
        0.530665, 0.885871, 0.0123631, 0.682109, 0.520907, 0.61765, 0.738911,
        0.477091, 0.593376, 0.660655, 0.400412, 0.550896, 0.523708, 0.42136,
        0.64639, 0.601285, 0.481359, 0.797072, 0.764672, 0.338876, 0.467272,
        0.647443, 0.162646, 0.0573917, 0.483445, 0.547367, 0.493703, 0.394239,
        0.820153, 0.871307, 0.316339, 0.42133, 0.896668, 0.306273, 0.2372,
        0.784422, 0.363343, 0.589802, 0.32816, 0.588256, 0.791288, 0.067439,
        0.716777, 0.292332, 0.319408, 0.864675, 0.322995, 0.341724, 0.70937,
        0.376654, 0.380775, 0.437585, 0.464387, 0.537448, 0.480948, 0.56575,
        0.741169, 0.650369, 0.352128, 0.440154, 0.577995, 0.0860078, 0.0550167,
        0.465321, 0.479133, 0.419873, 0.352859, 0.768945, 0.732637, 0.284935,
        0.438884, 0.732846, 0.484231, 0.24996, 0.656022, 0.656461, 0.413876,
        0.386614, 0.76103, 0.507542, 0.232667, 0.820783, 0.0637571, 0.021166,
        0.990439, 0.168898, 0.0900708, 0.758084, 0.352896, 0.210654, 0.351462,
        0.507414, 0.428506, 0.360611, 0.650142, 0.685266, 0.536067, 0.36538,
        0.413036, 0.508547, 0.00936967, 0.0526416, 0.447197, 0.410898, 0.346044,
        0.31148, 0.717737, 0.593968, 0.253531, 0.456438, 0.569025, 0.662189,
        0.262719, 0.527622, 0.94958, 0.237949, 0.445068, 0.933804, 0.223795,
        0.397895, 0.924789, 0.308829, 0.139718, 0.576134, 0.298417, 0.204672,
        0.534639, 0.280194, 0.318343, 0.462024, 0.406612, 0.503531, 0.530432,
        0.590885, 0.717326, 0.655248, 0.424742, 0.532242, 0.564865, 0.200196,
        0.280678, 0.438615, 0.35474, 0.459391, 0.265489, 0.477056, 0.612421,
        0.150863, 0.406004, 0.611351, 0.387231, 0.361845, 0.616633, 0.568817,
        0.384918, 0.637799, 0.613445, 0.398103, 0.649894, 0.638947, 0.553902,
        0.258269, 0.161828, 0.427935, 0.319274, 0.311195, 0.207493, 0.426031,
        0.572587, 0.305809, 0.578555, 0.700253, 0.531627, 0.749386, 0.77443,
        0.484104, 0.651447, 0.621183, 0.391023, 0.508714, 0.430033, 0.298582,
        0.572738, 0.219499, 0.236375, 0.630874, 0.048195, 0.35557, 0.653676,
        0.112274, 0.460971, 0.705645, 0.188053, 0.531887, 0.83053, 0.293086,
        0.572411, 0.901893, 0.353105, 0.478719, 0.345583, 0.288445, 0.373115,
        0.378028, 0.427302, 0.188308, 0.434805, 0.670301, 0.253353, 0.526846,
        0.685562, 0.418339, 0.632993, 0.609726, 0.458166, 0.618414, 0.542478,
        0.477133, 0.583714, 0.476661, 0.29887, 0.562377, 0.295839, 0.168751,
        0.550089, 0.139929, 0.327492, 0.59209, 0.133486, 0.481187, 0.647526,
        0.155137, 0.622262, 0.736545, 0.247023, 0.702877, 0.787413, 0.29953,
        0.24341, 0.417279, 0.685521, 0.226126, 0.408858, 0.713184, 0.19588,
        0.394121, 0.761592, 0.225071, 0.41177, 0.578614, 0.278036, 0.442374,
        0.30308, 0.389579, 0.509261, 0.396261, 0.510885, 0.582195, 0.550894,
        0.327381, 0.490162, 0.433345, 0.187655, 0.419685, 0.328863, 0.310593,
        0.478549, 0.302783, 0.461947, 0.51584, 0.296145, 0.68434, 0.499202,
        0.338108, 0.811422, 0.489694, 0.362087, 0.115431, 0.415056, 0.877619,
        0.154631, 0.379824, 0.843772, 0.223231, 0.318167, 0.784538, 0.234373,
        0.314811, 0.532135, 0.222532, 0.334776, 0.202464, 0.333324, 0.425583,
        0.33072, 0.464554, 0.528197, 0.535296, 0.372269, 0.459986, 0.463336,
        0.303998, 0.404409, 0.396939, 0.379814, 0.424645, 0.363914, 0.500517,
        0.417965, 0.357295, 0.73344, 0.343997, 0.41669, 0.866538, 0.30173,
        0.45063, 0.202114, 0.264994, 0.65976, 0.234124, 0.231059, 0.663772,
        0.29014, 0.171674, 0.670794, 0.318846, 0.254086, 0.606595, 0.336627,
        0.393218, 0.513907, 0.301731, 0.392854, 0.42653, 0.258057, 0.369241,
        0.340038, 0.44991, 0.513888, 0.278301, 0.615219, 0.619389, 0.223299,
        0.621274, 0.490018, 0.208711, 0.654706, 0.387712, 0.25873, 0.756582,
        0.35307, 0.470267, 0.814797, 0.333275, 0.591144, 0.288797, 0.114932,
        0.441901, 0.313616, 0.0822949, 0.483773, 0.35705, 0.0251804, 0.55705,
        0.403318, 0.193361, 0.681055, 0.45072, 0.451661, 0.82535, 0.270139,
        0.360126, 0.522341, 0.0515609, 0.210285, 0.14478, 0.52755, 0.56779,
        0.093265, 0.92644, 0.834369, 0.0496592, 0.862734, 0.55539, 0.0535081,
        0.808896, 0.357459, 0.160165, 0.779725, 0.362144, 0.523843, 0.763055,
        0.364821, 0.731659, 0.484536, 0.343428, 0.471467, 0.457236, 0.260657,
        0.52501, 0.409462, 0.115808, 0.61871, 0.387671, 0.231718, 0.65987,
        0.376273, 0.451933, 0.680014, 0.266391, 0.460296, 0.583782, 0.140094,
        0.43335, 0.468154, 0.392754, 0.669506, 0.405741, 0.62098, 0.82808,
        0.342252, 0.702599, 0.521167, 0.272301, 0.72906, 0.330968, 0.255662,
        0.617627, 0.432558, 0.372305, 0.553951, 0.490609, 0.438957, 0.680275,
        0.571925, 0.501034, 0.600857, 0.439019, 0.566247, 0.461874, 0.206435,
        0.68037, 0.372023, 0.270075, 0.638685, 0.301826, 0.452205, 0.534677,
        0.262642, 0.560466, 0.645222, 0.228627, 0.656414, 0.791527, 0.257958,
        0.771221, 0.718217, 0.31552, 0.821791, 0.634844, 0.542464, 0.486943,
        0.491093, 0.649225, 0.304478, 0.351159, 0.45553, 0.502972, 0.220766,
        0.344847, 0.616397, 0.146256, 0.745522, 0.64809, 0.510889, 0.64873,
        0.498473, 0.579992, 0.479344, 0.236644, 0.700923, 0.366808, 0.282861,
        0.631623, 0.27701, 0.452296, 0.486232, 0.261392, 0.593856, 0.665703,
        0.258138, 0.730769, 0.899318, 0.213026, 0.805126, 0.822375, 0.2137,
        0.819694, 0.732374, 0.489085, 0.475535, 0.564024, 0.622613, 0.295648,
        0.382992, 0.401498, 0.526443, 0.170254, 0.275146, 0.658326, 0.0486892,
        0.0457834, 0.787678, 0.663175, 0.17993, 0.7219, 0.59437, 0.414688,
        0.606787, 0.473961, 0.51223, 0.732484, 0.581171, 0.554887, 0.954503,
        0.77943, 0.459783, 0.979778, 0.566022, 0.34172, 0.972261, 0.284004,
        0.370952, 0.685892, 0.242341, 0.374854, 0.408401, 0.193533, 0.226783,
        0.184178, 0.101854, 0.24627, 0.0639145, 0.170491, 0.684648, 0.203548,
        0.639917, 0.93515, 0.283339, 0.90816, 0.138511, 0.716425, 0.678164,
        0.245857, 0.66815, 0.626455, 0.433711, 0.583667, 0.535964, 0.493074,
        0.679314, 0.630868, 0.501041, 0.847013, 0.79993, 0.446857, 0.914047,
        0.563479, 0.382315, 0.964302, 0.259442, 0.434202, 0.690901, 0.24902,
        0.458328, 0.427676, 0.225076, 0.31589, 0.225511, 0.119999, 0.325121,
        0.112612, 0.161402, 0.71352, 0.222875, 0.569003, 0.935462, 0.285882,
        0.801918, 0.416695, 0.502668, 0.723132, 0.443636, 0.5069, 0.722711,
        0.490781, 0.514307, 0.721973, 0.435606, 0.519806, 0.779958, 0.339502,
        0.524543, 0.861432, 0.408076, 0.716853, 0.555848, 0.504097, 0.940425,
        0.185755, 0.623952, 0.705927, 0.269057, 0.70875, 0.485502, 0.319706,
        0.583212, 0.349511, 0.174434, 0.561673, 0.258703, 0.134134, 0.800134,
        0.280854, 0.356262, 0.936397, 0.293511, 0.483193, 0.694879, 0.28891,
        0.7681, 0.641415, 0.345651, 0.818966, 0.547852, 0.444947, 0.907982,
        0.378138, 0.360298, 0.929048, 0.177964, 0.202073, 0.922934, 0.369296,
        0.519659, 0.548218, 0.625879, 0.916547, 0.112067, 0.813702, 0.720954,
        0.289094, 0.959173, 0.543329, 0.414335, 0.850533, 0.473511, 0.228869,
        0.798225, 0.404795, 0.106867, 0.886748, 0.338833, 0.143521, 0.937332,
        0.301141, 0.164467, 0.815943, 0.261243, 0.802579, 0.75255, 0.338519,
        0.816473, 0.641614, 0.473752, 0.840787, 0.558425, 0.485709, 0.75326,
        0.486336, 0.448357, 0.620998, 0.452304, 0.581538, 0.467379, 0.424616,
        0.743142, 0.3102, 0.58692, 0.633904, 0.301537, 0.734553, 0.545361,
        0.295396, 0.794157, 0.580998, 0.304381, 0.792835, 0.561136, 0.29601,
        0.639193, 0.402527, 0.244246, 0.551399, 0.311893, 0.214666, 0.937007,
        0.233576, 0.837057, 0.863686, 0.331387, 0.813979, 0.735375, 0.502557,
        0.773591, 0.738712, 0.61112, 0.577473, 0.794707, 0.694641, 0.319062,
        0.535312, 0.643418, 0.38654, 0.223352, 0.569737, 0.508333, 0.360138,
        0.546853, 0.313981, 0.509933, 0.547394, 0.176456, 0.737782, 0.688486,
        0.379893, 0.787444, 0.717478, 0.485153, 0.391639, 0.466221, 0.34497,
        0.165465, 0.322645, 0.264865, 0.966669, 0.299507, 0.776441, 0.877652,
        0.339989, 0.785181, 0.721873, 0.410833, 0.800478, 0.776405, 0.490512,
        0.625918, 0.915062, 0.573727, 0.375416, 0.578291, 0.585824, 0.474791,
        0.162281, 0.586067, 0.632477, 0.308455, 0.520064, 0.381298, 0.464959,
        0.495634, 0.19862, 0.683447, 0.720641, 0.426949, 0.733296, 0.792109,
        0.546757, 0.361548, 0.479734, 0.395264, 0.149121, 0.301234, 0.308697,
        0.950631, 0.412237, 0.668276, 0.843034, 0.356458, 0.743232, 0.654738,
        0.258843, 0.874405, 0.742802, 0.246895, 0.78648, 0.941409, 0.269214,
        0.610916, 0.601254, 0.468493, 0.647586, 0.171306, 0.697265, 0.719628,
        0.344321, 0.523406, 0.476052, 0.509808, 0.416977, 0.291335, 0.630132,
        0.715128, 0.459775, 0.654768, 0.825885, 0.544591, 0.440188, 0.468157,
        0.420344, 0.317571, 0.263741, 0.349346, 0.813592, 0.490177, 0.591749,
        0.74777, 0.383058, 0.708109, 0.632582, 0.1956, 0.91174, 0.7181,
        0.159092, 0.844851, 0.8839, 0.182963, 0.669753, 0.620864, 0.445203,
        0.662613, 0.286354, 0.747172, 0.683465, 0.403892, 0.550087, 0.542958,
        0.515033, 0.421903, 0.430845, 0.587788, 0.707126, 0.489095, 0.613344,
        0.805958, 0.514667, 0.520904, 0.43881, 0.458542, 0.468081, 0.229012,
        0.426471, 0.434549, 0.498534, 0.578495, 0.531216, 0.429922, 0.686639,
        0.700383, 0.30985, 0.875891, 0.711202, 0.382914, 0.698839, 0.658681,
        0.533233, 0.375266, 0.633765, 0.609996, 0.362104, 0.61345, 0.674499,
        0.400677, 0.510875, 0.623447, 0.55417, 0.441007, 0.593996, 0.663944,
        0.567386, 0.694145, 0.511401, 0.646126, 0.678623, 0.429228, 0.605769,
        0.373922, 0.522976, 0.582708, 0.199808, 0.576546, 0.0555074, 0.506892,
        0.565241, 0.314663, 0.476786, 0.665169, 0.768185, 0.4241, 0.840043,
        0.704304, 0.606737, 0.552828, 0.433463, 0.883503, 0.0807784, 0.646667,
        0.774788, 0.0615947, 0.940546, 0.601826, 0.117889, 0.617857, 0.696808,
        0.565382, 0.366981, 0.766088, 0.897043, 0.546984, 0.681164, 0.533708,
        0.678907, 0.551288, 0.343789, 0.690635, 0.309035, 0.587409, 0.697336,
        0.170604, 0.726621, 0.281613, 0.328835, 0.589203, 0.373649, 0.363073,
        0.670092, 0.534713, 0.42299, 0.811647, 0.556594, 0.551088, 0.537557,
        0.522801, 0.706459, 0.0972087, 0.538583, 0.74625, 0.0971787, 0.562627,
        0.766779, 0.170535, 0.469515, 0.63935, 0.562376, 0.417981, 0.533251,
        0.861871, 0.615918, 0.555136, 0.607292, 0.75222, 0.553922, 0.454692,
        0.734438, 0.494964, 0.557036, 0.724277, 0.461273, 0.615519, 0.507718,
        0.150778, 0.613165, 0.432636, 0.249361, 0.675015, 0.301242, 0.42188,
        0.783251, 0.408884, 0.495439, 0.522285, 0.612139, 0.529414, 0.113639,
        0.430499, 0.717713, 0.132763, 0.184709, 0.931731, 0.223181, 0.321173,
        0.581892, 0.559369, 0.468981, 0.300414, 0.826699, 0.684852, 0.429108,
        0.680876, 0.825533, 0.556557, 0.565594, 0.778242, 0.680893, 0.526663,
        0.751218, 0.751942, 0.504416, 0.583086, 0.0914261, 0.621153, 0.452298,
        0.211456, 0.676656, 0.223418, 0.421509, 0.773785, 0.359647, 0.476889,
        0.517195, 0.641919, 0.470399, 0.119116, 0.394471, 0.7082, 0.144624,
        0.0587368, 0.986715, 0.24073, 0.271726, 0.562739, 0.558367, 0.485981,
        0.222802, 0.814975, 0.70783, 0.387099, 0.705405, 0.849971, 0.557435,
        0.602562, 0.792843, 0.742869, 0.516539, 0.760198, 0.848832, 0.467382
    ],
    add_quant_variations=False,
)

test(
    name="align_corners_2x2_to_1x1",
    input0=Input("input0", "TENSOR_FLOAT32", "{1, 2, 2, 1}"),
    output_width=Int32Scalar("output_width", 1),
    output_height=Int32Scalar("output_height", 1),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", True),
    half_pixel_centers=BoolScalar("half_pixel_centers", False),
    output0=Output("output0", "TENSOR_FLOAT32", "{1, 1, 1, 1}"),
    input0_data=[1, 2, 3, 4],
    output_data=[1],
)

test(
    name="align_corners_2x2_to_3x3",
    input0=Input("input0", "TENSOR_FLOAT32", "{1, 2, 2, 1}"),
    output_width=Int32Scalar("output_width", 3),
    output_height=Int32Scalar("output_height", 3),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", True),
    half_pixel_centers=BoolScalar("half_pixel_centers", False),
    output0=Output("output0", "TENSOR_FLOAT32", "{1, 3, 3, 1}"),
    input0_data=[1, 2, 3, 4],
    output_data=[1, 1.5, 2, 2, 2.5, 3, 3, 3.5, 4],
)

test(
    name="align_corners_3x3_to_2x2",
    input0=Input("input0", "TENSOR_FLOAT32", "{1, 3, 3, 1}"),
    output_width=Int32Scalar("output_width", 2),
    output_height=Int32Scalar("output_height", 2),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", True),
    half_pixel_centers=BoolScalar("half_pixel_centers", False),
    output0=Output("output0", "TENSOR_FLOAT32", "{1, 2, 2, 1}"),
    input0_data=[1, 2, 3, 4, 5, 6, 7, 8, 9],
    output_data=[1, 3, 7, 9],
)

test(
    name="align_corners_4x4_to_3x3",
    input0=Input("input0", "TENSOR_FLOAT32", "{1, 4, 4, 1}"),
    output_width=Int32Scalar("output_width", 3),
    output_height=Int32Scalar("output_height", 3),
    layout=BoolScalar("layout", False),
    align_corners=BoolScalar("align_corners", True),
    half_pixel_centers=BoolScalar("half_pixel_centers", False),
    output0=Output("output0", "TENSOR_FLOAT32", "{1, 3, 3, 1}"),
    input0_data=[1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16],
    output_data=[1, 2.5, 4, 7, 8.5, 10, 13, 14.5, 16],
)

# Test from version 1.2 to test default arguments removal
i2 = Input("op1", "TENSOR_FLOAT32", "{1, 2, 2, 2}")
o2 = Output("op4", "TENSOR_FLOAT32", "{1, 3, 3, 2}")
layout = BoolScalar("layout", False)  # NHWC
align_corners = BoolScalar("align_corners", False)
half_pixel_centers = BoolScalar("half_pixel_centers", False)
model_shape = Model("shape").Operation("RESIZE_BILINEAR", i2, 3, 3, layout,
                                       align_corners, half_pixel_centers).To(o2)
model_scale = Model("scale").Operation("RESIZE_BILINEAR", i2, 1.6, 1.6, layout,
                                       align_corners, half_pixel_centers).To(o2)

# Additional data type
quant8 = DataTypeConverter().Identify({
    i2: ("TENSOR_QUANT8_ASYMM", 0.25, 0),
    o2: ("TENSOR_QUANT8_ASYMM", 0.25, 0)
})

values = {
    i2: [3, 4, 6, 10, 9, 10, 12, 16],
    o2: [3, 4, 5, 8, 6, 10, 7, 8, 9, 12, 10, 14, 9, 10, 11, 14, 12, 16]
}

# Instantiate an example
Example(
    values, model=model_shape,
    name="default_values").AddNchw(i2, o2, layout).AddVariations(
        "relaxed", "float16", quant8)
Example(
    values, model=model_scale,
    name="default_values").AddNchw(i2, o2, layout).AddVariations(
        "relaxed", "float16", quant8)

Example.SetVersion(
    "V1_2",
    "resize_bilinear_v1_3_scale_default_values_nchw",
    "resize_bilinear_v1_3_scale_default_values_nchw_all_inputs_as_internal",
    "resize_bilinear_v1_3_scale_default_values_nchw_float16",
    "resize_bilinear_v1_3_scale_default_values_nchw_float16_all_inputs_as_internal",
    "resize_bilinear_v1_3_scale_default_values_nchw_quant8",
    "resize_bilinear_v1_3_scale_default_values_nchw_quant8_all_inputs_as_internal",
    "resize_bilinear_v1_3_scale_default_values_nhwc",
    "resize_bilinear_v1_3_scale_default_values_nhwc_all_inputs_as_internal",
    "resize_bilinear_v1_3_scale_default_values_nhwc_float16",
    "resize_bilinear_v1_3_scale_default_values_nhwc_float16_all_inputs_as_internal",
    "resize_bilinear_v1_3_scale_default_values_nhwc_quant8",
    "resize_bilinear_v1_3_scale_default_values_nhwc_quant8_all_inputs_as_internal",
    "resize_bilinear_v1_3_shape_default_values_nchw",
    "resize_bilinear_v1_3_shape_default_values_nchw_all_inputs_as_internal",
    "resize_bilinear_v1_3_shape_default_values_nchw_float16",
    "resize_bilinear_v1_3_shape_default_values_nchw_float16_all_inputs_as_internal",
    "resize_bilinear_v1_3_shape_default_values_nchw_quant8",
    "resize_bilinear_v1_3_shape_default_values_nchw_quant8_all_inputs_as_internal",
    "resize_bilinear_v1_3_shape_default_values_nhwc_float16",
    "resize_bilinear_v1_3_shape_default_values_nhwc_float16_all_inputs_as_internal",
    "resize_bilinear_v1_3_shape_default_values_nhwc_quant8",
    "resize_bilinear_v1_3_shape_default_values_nhwc_quant8_all_inputs_as_internal",
)

Example.SetVersion(
    "V1_0",
    "resize_bilinear_v1_3_shape_default_values_nhwc",
    "resize_bilinear_v1_3_shape_default_values_nhwc_all_inputs_as_internal",
)
