// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * f_iap.c -- Alpine USB peripheral source/sink configuration driver
 *
 * Copyright (C) 2016-2018 ALPINE ELECTRONICS, INC.
 * Copyright (C) 2019 ALPSALPINE CO.,LTD.
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/usb/composite.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/sched.h>

#include "apnusb.h"
#include "u_iap.h"

/*
 * This function packages
 *
 *
 */

#define ALPINE_IAP_NAME				"iAP Interface"

#define ALPINE_IAP_DEVICE_NUM			(1)

#define ALPINE_IAP_MAX_READ_BUFF		(1)
#define ALPINE_IAP_MAX_WRITE_BUFF		(10)

#define ALPINE_IAP_OPEN_MAX			(8)

/*-------------------------------------------------------------------------*/

struct apn_event {
	int			event;
	int			status;
	struct list_head	list;
};

struct apn_file {
	struct file		*fd_p;
	bool			is_nonblock;
	struct list_head	event_list;
	wait_queue_head_t	event_queue;
	spinlock_t		spinlock;
};

struct apn_request {
	struct usb_request	*req_p;
	unsigned int		offset;
	unsigned int		leftsize;
	struct apn_file		*apnfd_p;
};

struct apn_device {
	struct device		dev;
	struct cdev		cdev;
};

struct f_alpine_iap {
	struct usb_function	function;

	struct usb_ep		*ep_in_p;
	struct usb_ep		*ep_out_p;

	int			open_num;
	bool			is_connect;
	bool			is_aborted;	/* Not update to true. */

	struct usb_request	*req_in_a[ALPINE_IAP_MAX_WRITE_BUFF];
	struct usb_request	*req_out_a[ALPINE_IAP_MAX_READ_BUFF];

	struct apn_device	*apn_dev_p;

	struct apn_file		*apnfd_p[ALPINE_IAP_OPEN_MAX];

	/* read buff */
	struct list_head	read_buff_queue;
	wait_queue_head_t	read_queue;

	/* write buff */
	struct list_head	write_buff_pool;
	wait_queue_head_t	write_queue;

	/* spinlock */
	spinlock_t		spinlock;
};

static inline struct f_alpine_iap *func_to_alpiap(struct usb_function *f)
{
	return container_of(f, struct f_alpine_iap, function);
}

/*-------------------------------------------------------------------------*/

static struct usb_interface_assoc_descriptor alpine_iap_interface_assoc_desc = {
	.bLength		= USB_DT_INTERFACE_ASSOCIATION_SIZE,
	.bDescriptorType	= USB_DT_INTERFACE_ASSOCIATION,

	/* .bFirstInterface = DYNAMIC, */
	.bInterfaceCount	= 1,
	.bFunctionClass		= USB_CLASS_VENDOR_SPEC,
	.bFunctionSubClass	= 0xF0,
	.bFunctionProtocol	= 0,
	/* .iFunction = DYNAMIC */
};

/* interface descriptor: */
static struct usb_interface_descriptor alpine_iap_interface_desc = {
	.bLength		= USB_DT_INTERFACE_SIZE,
	.bDescriptorType	= USB_DT_INTERFACE,
	/* .bInterfaceNumber = DYNAMIC */
	.bAlternateSetting	= 0,
	.bNumEndpoints		= 2,
	.bInterfaceClass	= USB_CLASS_VENDOR_SPEC,
	.bInterfaceSubClass	= 0xF0,	/* MFi accessory */
	.bInterfaceProtocol	= 0,
	/* .iInterface = DYNAMIC */
};

/* full speed support */
static struct usb_endpoint_descriptor alpine_iap_ep_fs_in_desc = {
	.bLength		= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType	= USB_DT_ENDPOINT,

	.bEndpointAddress	= USB_DIR_IN,
	.bmAttributes		= USB_ENDPOINT_XFER_BULK,
};

static struct usb_endpoint_descriptor alpine_iap_ep_fs_out_desc = {
	.bLength		= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType	= USB_DT_ENDPOINT,

	.bEndpointAddress	= USB_DIR_OUT,
	.bmAttributes		= USB_ENDPOINT_XFER_BULK,
};

static struct usb_descriptor_header *alpine_iap_fs_descs[] = {
	(struct usb_descriptor_header *) &alpine_iap_interface_assoc_desc,
	(struct usb_descriptor_header *) &alpine_iap_interface_desc,
	(struct usb_descriptor_header *) &alpine_iap_ep_fs_in_desc,
	(struct usb_descriptor_header *) &alpine_iap_ep_fs_out_desc,
	NULL,
};

/* high speed support */
static struct usb_endpoint_descriptor alpine_iap_ep_hs_in_desc = {
	.bLength		= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType	= USB_DT_ENDPOINT,

	.bEndpointAddress	= USB_DIR_IN,
	.bmAttributes		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize		= cpu_to_le16(512),
};

static struct usb_endpoint_descriptor alpine_iap_ep_hs_out_desc = {
	.bLength		= USB_DT_ENDPOINT_SIZE,
	.bDescriptorType	= USB_DT_ENDPOINT,

	.bEndpointAddress	= USB_DIR_OUT,
	.bmAttributes		= USB_ENDPOINT_XFER_BULK,
	.wMaxPacketSize		= cpu_to_le16(512),
};

static struct usb_descriptor_header *alpine_iap_hs_descs[] = {
	(struct usb_descriptor_header *) &alpine_iap_interface_assoc_desc,
	(struct usb_descriptor_header *) &alpine_iap_interface_desc,
	(struct usb_descriptor_header *) &alpine_iap_ep_hs_in_desc,
	(struct usb_descriptor_header *) &alpine_iap_ep_hs_out_desc,
	NULL,
};

/* function-specific strings: */

#define STRING_INTERFACE_IDX	0

static struct usb_string strings_alpine_iap_defs[] = {
	[STRING_INTERFACE_IDX].s = ALPINE_IAP_NAME,
	{  }			/* end of list */
};

static struct usb_gadget_strings stringtab_alpine_iap_table = {
	.language	= 0x0409,	/* en-us */
	.strings	= strings_alpine_iap_defs,
};

static struct usb_gadget_strings *alpine_iap_strings[] = {
	&stringtab_alpine_iap_table,
	NULL,
};

static inline struct f_alpine_opts *to_f_alpine_opts(struct config_item *item)
{
	return container_of(to_config_group(item), struct f_alpine_opts,
				func_inst.group);
}

static void alpine_attr_release(struct config_item *item)
{
	struct f_alpine_opts *opts = to_f_alpine_opts(item);

	usb_put_function_instance(&opts->func_inst);
}

/*
 * Note: following f_alpine_opts_attr_show and
 * f_alpine_attr_store are created by CONFIG_ATTR_STRUCT macro
 */
static struct configfs_item_operations alpine_item_ops = {
	.release	= alpine_attr_release,
};

static struct configfs_attribute *alpine_attrs[] = {
	NULL,
};

static struct config_item_type alpine_func_type = {
	.ct_item_ops	= &alpine_item_ops,
	.ct_attrs	= alpine_attrs,
	.ct_owner	= THIS_MODULE,
};

/*-------------------------------------------------------------------------*/

static int			 apn_iap_major;
static int			 apn_iap_minor;
static struct class		*apn_iap_class_p;

/*-------------------------------------------------------------------------*/

#define READ_COND_WAIT		((iap_p->is_connect != false) \
				  && (iap_p->is_aborted == false) \
				  && (list_empty(&iap_p->read_buff_queue) != 0))
#define READ_COND_ENABLE	((iap_p->is_connect == false) \
				  || (iap_p->is_aborted != false) \
				  || (list_empty(&iap_p->read_buff_queue) == 0))
#define WRITE_COND_WAIT		((iap_p->is_connect != false) \
				  && (iap_p->is_aborted == false) \
				  && (list_empty(&iap_p->write_buff_pool) != 0))
#define WRITE_COND_ENABLE	((iap_p->is_connect == false) \
				  || (iap_p->is_aborted != false) \
				  || (list_empty(&iap_p->write_buff_pool) == 0))

/*-------------------------------------------------------------------------*/

static struct apn_file **get_apn_file_pp(struct f_alpine_iap *iap_p,
					struct file *fd_p)
{
	int				index_local;
	unsigned long			flags_local, flags_apnfd;
	struct apn_file			**apnfd_pp;

	apnfd_pp = NULL;

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	for (index_local = 0; index_local < ALPINE_IAP_OPEN_MAX; index_local++) {
		if (iap_p->apnfd_p[index_local] != NULL) {
			spin_lock_irqsave(
				&iap_p->apnfd_p[index_local]->spinlock, flags_apnfd);
			if (iap_p->apnfd_p[index_local]->fd_p == fd_p) {
				apnfd_pp = &iap_p->apnfd_p[index_local];
				spin_unlock_irqrestore(
					&iap_p->apnfd_p[index_local]->spinlock,
					flags_apnfd);
				break;
			}
			spin_unlock_irqrestore(
				&iap_p->apnfd_p[index_local]->spinlock, flags_apnfd);
		}
	}
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	return apnfd_pp;
}

static struct apn_file *get_apn_file_p(struct f_alpine_iap *iap_p,
					struct file *fd_p)
{
	struct apn_file			*apnfd_p;
	struct apn_file			**apnfd_pp;

	apnfd_pp = get_apn_file_pp(iap_p, fd_p);

	if (apnfd_pp == NULL)
		apnfd_p = NULL;
	else
		apnfd_p = *apnfd_pp;

	return apnfd_p;
}

static int check_connection(struct f_alpine_iap *iap_p)
{
	int	result;

	result = 0;

	if ((bool)unlikely(iap_p->is_connect == false)) {
		/* disconnected */
		result = -ENOTCONN;
	} else if ((bool)unlikely(iap_p->is_aborted != false)) {
		/* aborted */
		result = -ECANCELED;
	} else {
		/* nothing to do */
	}

	return result;
}

static int get_read_buffer(struct f_alpine_iap *iap_p,
			   bool is_nonblock,
			   struct usb_request **req_pp)
{
	unsigned long		flags_local;
	int			result;

	result		= 0;

	/* check read buffer queue */
	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	while (READ_COND_WAIT) {
		spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
		if (is_nonblock) {
			result = -EAGAIN;
			return result;
		}

		/* wait for queued */
		if (wait_event_interruptible_exclusive(
				iap_p->read_queue,
				!READ_COND_WAIT) != 0) {
			result = -ERESTARTSYS;
			return result;
		}
		spin_lock_irqsave(&iap_p->spinlock, flags_local);
	}

	/* get read buffer */
	if ((bool)likely(list_empty(&iap_p->read_buff_queue) == 0)) {
		*req_pp = list_first_entry(
			&iap_p->read_buff_queue, struct usb_request, list);
	} else {
		*req_pp = NULL;
	}

	result = check_connection(iap_p);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	return result;
}

static ssize_t f_alpine_iap_read(struct file *fd_p,
				 char __user *buf_p,
				 size_t count,
				 loff_t *off_p)
{
	int				result;
	int				cpyrslt;
	size_t				read_size;
	struct f_alpine_iap		*iap_p;
	struct apn_file			*apnfd_p;
	struct usb_request		*req_p;
	struct apn_request		*apn_req_p;
	unsigned long			flags_local;

	result		= 0;
	read_size	= 0;

	iap_p		= fd_p->private_data;

	if ((bool)unlikely(iap_p == NULL)) {
		result = -EPERM;
		goto fail;
	}

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	result = check_connection(iap_p);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
	if ((bool)unlikely(result != 0))
		goto fail;

	apnfd_p = get_apn_file_p(iap_p, fd_p);
	if (apnfd_p == NULL) {
		result = -EPERM;
		goto fail;
	}

	result = get_read_buffer(
		iap_p, ((fd_p->f_flags & O_NONBLOCK) == O_NONBLOCK), &req_p);
	if ((bool)unlikely(result != 0))
		goto fail;

	if ((bool)unlikely(req_p == NULL)) {
		/* no buffer */
		result = -EINVAL;
		goto fail;
	}

	if ((bool)unlikely(req_p->status != 0)) {
		if (req_p->status != -ESHUTDOWN)
			result = -ECOMM;
		goto fail;
	}

	apn_req_p = (struct apn_request *)req_p->context;

	read_size = min_t(unsigned int, count, apn_req_p->leftsize);
	cpyrslt = copy_to_user(
				buf_p,
				(((char *)req_p->buf) + apn_req_p->offset),
				read_size);

	if ((bool)unlikely(cpyrslt != 0)) {
		/* copy failure */
		result = -EFAULT;
		goto fail;
	}

	/* copy success */
	apn_req_p->leftsize -= read_size;
	apn_req_p->apnfd_p = apnfd_p;
	if ((bool)likely(apn_req_p->leftsize == 0)) {
		list_del(&req_p->list);
		/* set read buffer to ep */
		req_p->status		= 0;
		req_p->zero		= 0;
		req_p->length		= APN_USB_BUFFLEN;

		result = usb_ep_queue(iap_p->ep_out_p, req_p, GFP_ATOMIC);
		if ((bool)likely(result >= 0)) {
			/* success */
			result = read_size;
		} else {
			/* failure */
		}
	} else {
		apn_req_p->offset += read_size;

		result = read_size;
	}

fail:
/*
 *	if (likely(iap_p) && (result != -EAGAIN)) {
 *		DBG(iap_p->function.config->cdev, "%s called %d:%d:%d\n",
 *			__func__, result, count, read_size);
 *	}
 */
	return result;
}

static int get_write_buffer(struct f_alpine_iap *iap_p,
			    bool is_nonblock,
			    struct usb_request **req_pp)
{
	unsigned long		flags_local;
	int			result;

	result		= 0;

	/* check write buffer spool */
	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	while (WRITE_COND_WAIT) {
		spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
		if (is_nonblock) {
			result = -EAGAIN;
			return result;
		}

		/* wait for spooled */
		if (wait_event_interruptible_exclusive(
				iap_p->write_queue,
				!WRITE_COND_WAIT) != 0) {
			result = -ERESTARTSYS;
			return result;
		}
		spin_lock_irqsave(&iap_p->spinlock, flags_local);
	}

	/* get write buffer */
	if ((bool)likely(list_empty(&iap_p->write_buff_pool) == 0)) {
		*req_pp = list_first_entry(
			&iap_p->write_buff_pool, struct usb_request, list);
	} else {
		*req_pp = NULL;
	}

	result = check_connection(iap_p);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	return result;
}

static ssize_t f_alpine_iap_write(struct file *fd_p,
				  const char __user *buf_p,
				  size_t count, loff_t *off_p)
{
	int				result;
	int				cpyrslt;
	size_t				write_length;
	struct f_alpine_iap		*iap_p;
	struct apn_file			*apnfd_p;
	struct usb_request		*req_p;
	struct apn_request		*apn_req_p;
	unsigned long			flags_local;

	result		= 0;
	write_length	= 0;

	iap_p		= fd_p->private_data;

	if ((unlikely(iap_p == NULL)) || (unlikely(iap_p->ep_in_p == NULL))) {
		result = -EPERM;
		goto fail;
	}

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	result = check_connection(iap_p);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
	if ((bool)unlikely(result != 0))
		goto fail;

	apnfd_p = get_apn_file_p(iap_p, fd_p);
	if (apnfd_p == NULL) {
		result = -EPERM;
		goto fail;
	}

	result = get_write_buffer(
		iap_p, ((fd_p->f_flags & O_NONBLOCK) == O_NONBLOCK), &req_p);
	if ((bool)unlikely(result != 0))
		goto fail;

	if ((bool)unlikely(req_p == NULL)) {
		/* no buffer */
		result = -EINVAL;
		goto fail;
	}

	apn_req_p = (struct apn_request *)req_p->context;

	write_length = min_t(unsigned int, count, APN_USB_BUFFLEN);
	cpyrslt = copy_from_user(req_p->buf, buf_p, write_length);

	if ((bool)unlikely(cpyrslt != 0)) {
		/* copy fail */
		result = -EINVAL;
		goto fail;
	}

	/* copy success */
	apn_req_p->apnfd_p = apnfd_p;

	spin_lock_irqsave(&iap_p->spinlock, flags_local);

	list_del(&req_p->list);

	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	req_p->status	= 0;
	req_p->zero	= ((count <= APN_USB_BUFFLEN) &&
		((count % iap_p->ep_in_p->maxpacket) == 0));
	req_p->length	= write_length;

	result = usb_ep_queue(iap_p->ep_in_p, req_p, GFP_ATOMIC);
	if ((bool)likely(result >= 0)) {
		/* success */
		result = write_length;
	} else {
		/* failure */
	}

fail:
/*
 *	if (likely(iap_p)) {
 *		DBG(iap_p->function.config->cdev, "%s called %d\n", __func__,
 *			result);
 *	}
 */
	return result;
}

static int f_alpine_iap_open(struct inode *inode_p, struct file *fd_p)
{
	int				result;
	int				index_local;
	struct f_alpine_iap		*iap_p;
	struct apn_device		*apn_dev_p;
	struct apn_event		*event_p;
	struct apn_event		*next_p;
	struct apn_file			*apnfd_p;
	unsigned long			flags_local;

	apn_dev_p = container_of(inode_p->i_cdev, struct apn_device, cdev);
	iap_p = dev_get_drvdata(&apn_dev_p->dev);

	spin_lock_irqsave(&iap_p->spinlock, flags_local);

	if ((bool)unlikely((iap_p->open_num < 0)
		|| (iap_p->open_num >= ALPINE_IAP_OPEN_MAX))) {
		/* already all open */
		result = -EBUSY;
		spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
		goto fail;
	}

	/* get apn_file */
	for (index_local = 0; index_local < ALPINE_IAP_OPEN_MAX; index_local++) {
		if (iap_p->apnfd_p[index_local] == NULL) {
			spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
			apnfd_p = kzalloc(sizeof(struct apn_file), GFP_KERNEL);
			if ((bool)unlikely(apnfd_p == NULL)) {
				result = -ENOMEM;
				goto fail;
			}

			spin_lock_irqsave(&iap_p->spinlock, flags_local);
			iap_p->apnfd_p[index_local] = apnfd_p;
			break;
		}
	}

	if (index_local == ALPINE_IAP_OPEN_MAX) {
		result = -EBUSY;
		spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
		goto fail;
	}

	/* new open */
	fd_p->private_data = iap_p;
	iap_p->open_num++;
	get_device(&apn_dev_p->dev);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	spin_lock_init(&apnfd_p->spinlock);

	spin_lock_irqsave(&apnfd_p->spinlock, flags_local);
	apnfd_p->fd_p = fd_p;

	if ((fd_p->f_flags & O_NONBLOCK) == O_NONBLOCK)
		apnfd_p->is_nonblock = true;
	else
		apnfd_p->is_nonblock = false;

	/* init event list */
	INIT_LIST_HEAD(&apnfd_p->event_list);

	/* purge event list */
	list_for_each_entry_safe(event_p, next_p, &apnfd_p->event_list, list) {
		list_del(&event_p->list);
		kfree(event_p);
	}
	spin_unlock_irqrestore(&apnfd_p->spinlock, flags_local);

	/* init event queue */
	init_waitqueue_head(&apnfd_p->event_queue);

	result = 0;

fail:
	pr_info("%s called %d\n", __func__, result);
	return result;
}

static int f_alpine_iap_close(struct inode *inode_p, struct file *fd_p)
{
	int				result;
	struct f_alpine_iap		*iap_p;
	struct apn_event		*event_p;
	struct apn_event		*next_p;
	struct apn_file			*apnfd_p;
	struct apn_file			**apnfd_pp;
	unsigned long			flags_local;

	result = 0;
	apnfd_p = NULL;
	apnfd_pp = NULL;

	iap_p = fd_p->private_data;
	if ((bool)unlikely(iap_p == NULL)) {
		result = -EPERM;
		goto fail;
	}

	apnfd_pp = get_apn_file_pp(iap_p, fd_p);
	if (apnfd_pp == NULL) {
		result = -ENOENT;
		goto fail;
	}
	apnfd_p = *apnfd_pp;

	spin_lock_irqsave(&apnfd_p->spinlock, flags_local);
	/* purge event list */
	list_for_each_entry_safe(event_p, next_p, &apnfd_p->event_list, list) {
		list_del(&event_p->list);
		kfree(event_p);
	}
	spin_unlock_irqrestore(&apnfd_p->spinlock, flags_local);

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	kfree(apnfd_p);
	*apnfd_pp = NULL;
	put_device(&iap_p->apn_dev_p->dev);
	iap_p->open_num--;
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

fail:
	fd_p->private_data = NULL;
	pr_info("%s called %d\n", __func__, result);
	return result;
}

const struct file_operations f_alpine_iap_fops = {
	.owner			= THIS_MODULE,
	.open			= f_alpine_iap_open,
	.release		= f_alpine_iap_close,
	.write			= f_alpine_iap_write,
	.read			= f_alpine_iap_read,
};

/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/

static struct usb_request *alloc_ep_req(struct usb_ep *ep_p)
{
	struct usb_request		*req_p;
	struct apn_request		*apn_req_p;

	req_p = usb_ep_alloc_request(ep_p, GFP_ATOMIC);
	if ((bool)likely(req_p != NULL)) {
		req_p->length = APN_USB_BUFFLEN;
		req_p->buf = kmalloc(APN_USB_BUFFLEN, GFP_ATOMIC);
		if ((bool)likely(req_p->buf != NULL)) {
			apn_req_p = kmalloc(
				sizeof(struct apn_request), GFP_ATOMIC);
			if ((bool)likely(apn_req_p != NULL)) {
				req_p->context = apn_req_p;
				apn_req_p->req_p = req_p;
				apn_req_p->offset = 0;
				apn_req_p->leftsize = 0;
				apn_req_p->apnfd_p = NULL;
			} else {
				kfree(req_p->buf);
				usb_ep_free_request(ep_p, req_p);
				req_p = NULL;
			}
		} else {
			usb_ep_free_request(ep_p, req_p);
			req_p = NULL;
		}
	} else {
		/* nothing to do */
	}

	return req_p;
}

static void free_ep_req(struct usb_ep *ep_p, struct usb_request *req_p)
{
	kfree(req_p->context);
	kfree(req_p->buf);
	usb_ep_free_request(ep_p, req_p);
}
/*-------------------------------------------------------------------------*/

static void alpine_iap_in_complete(struct usb_ep *ep_p,
				   struct usb_request *req_p)
{
/*	struct usb_composite_dev	*cdev_p;	*/
	struct f_alpine_iap		*iap_p;
	unsigned long			flags_local;

	iap_p		= ep_p->driver_data;

	if ((bool)unlikely(iap_p == NULL))
		return;

	if (iap_p->is_connect == false)
		return;

	/* BULK IN */

	/* add to pool */
	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	list_add_tail(&req_p->list, &iap_p->write_buff_pool);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
	wake_up(&iap_p->write_queue);

/*
 *	cdev_p = iap_p->function.config->cdev;
 *	DBG(cdev_p, "%s complete %d\n", ep_p->name, status);
 */
}

static void alpine_iap_out_complete(struct usb_ep *ep_p,
				    struct usb_request *req_p)
{
/*	struct usb_composite_dev	*cdev_p;	*/
	struct f_alpine_iap		*iap_p;
	struct apn_request		*apn_req_p;
	unsigned long			flags_local;

	iap_p		= ep_p->driver_data;

	if ((bool)unlikely(iap_p == NULL))
		return;

	if (iap_p->is_connect == false)
		return;

	/* BULK_OUT */

	apn_req_p = (struct apn_request *)req_p->context;
	apn_req_p->offset	= 0;
	apn_req_p->leftsize = req_p->actual;

	/* add to queue */
	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	list_add_tail(&req_p->list, &iap_p->read_buff_queue);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
	wake_up(&iap_p->read_queue);

/*
 *	cdev_p = iap_p->function.config->cdev;
 *	DBG(cdev_p, "%s complete %d\n", ep_p->name, status);
 */
}

/*-------------------------------------------------------------------------*/

static void set_connect_event(struct f_alpine_iap *iap_p, int event)
{
	int				index_local;
	struct apn_event		*event_p;
	unsigned long			flags_local;

	event_p = NULL;

	for (index_local = 0; index_local < ALPINE_IAP_OPEN_MAX; index_local++) {
		spin_lock_irqsave(&iap_p->spinlock, flags_local);
		if (iap_p->apnfd_p[index_local] != NULL) {
			spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
			event_p = kzalloc(sizeof(struct apn_event), GFP_ATOMIC);
			if ((bool)likely(event_p != NULL)) {
				event_p->event = event;
				spin_lock_irqsave(
					&iap_p->apnfd_p[index_local]->spinlock,
					flags_local);
				list_add_tail(&event_p->list,
					&iap_p->apnfd_p[index_local]->event_list);
				spin_unlock_irqrestore(
					&iap_p->apnfd_p[index_local]->spinlock,
					flags_local);
				wake_up(&iap_p->apnfd_p[index_local]->event_queue);
			}
			event_p = NULL;
		} else {
			spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
		}
	}
}

static int allocate_write_buffer(struct f_alpine_iap *iap_p)
{
	int				result;
	int				index_local;
	struct usb_request		*req_p;
	unsigned long			flags_local;

	result		= 0;
	req_p		= NULL;

	for (index_local = 0; index_local < ALPINE_IAP_MAX_WRITE_BUFF; index_local++) {
		req_p = alloc_ep_req(iap_p->ep_in_p);
		if ((bool)likely(req_p != NULL)) {
			req_p->complete = alpine_iap_in_complete;
			spin_lock_irqsave(&iap_p->spinlock, flags_local);
			iap_p->req_in_a[index_local] = req_p;
			list_add_tail(&req_p->list, &iap_p->write_buff_pool);
			spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
		} else {
			result = -ENOMEM;
			return result;
		}
	}

	return result;
}

static int allocate_read_buffer(struct f_alpine_iap *iap_p)
{
	int				result;
	int				index_local;
	struct usb_request		*req_p;
	unsigned long			flags_local;

	result		= 0;
	req_p		= NULL;

	for (index_local = 0; index_local < ALPINE_IAP_MAX_READ_BUFF; index_local++) {
		req_p = alloc_ep_req(iap_p->ep_out_p);
		if ((bool)likely(req_p != NULL)) {
			req_p->complete = alpine_iap_out_complete;
			req_p->length	= APN_USB_BUFFLEN;
			spin_lock_irqsave(&iap_p->spinlock, flags_local);
			iap_p->req_out_a[index_local] = req_p;
			spin_unlock_irqrestore(&iap_p->spinlock, flags_local);
			result = usb_ep_queue(
				iap_p->ep_out_p, req_p, GFP_ATOMIC);

			if ((bool)unlikely(result != 0))
				return result;
		} else {
			result = -ENOMEM;
			return result;
		}
	}

	return result;
}

static int enable_alpine_iap(struct usb_composite_dev *cdev_p,
			     struct f_alpine_iap *iap_p)
{
	int				result;
	int				index_local;
	struct usb_ep			*ep_in_p;
	struct usb_ep			*ep_out_p;
	unsigned long			flags_local;

	result		= 0;
	ep_in_p		= NULL;
	ep_out_p	= NULL;

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	memset(iap_p->req_in_a,  0x00,
		(sizeof(struct usb_request *) * ALPINE_IAP_MAX_WRITE_BUFF));
	memset(iap_p->req_out_a, 0x00,
		(sizeof(struct usb_request *) * ALPINE_IAP_MAX_READ_BUFF));
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	/* one bulk endpoint writes zeroes IN (to the host) */
	ep_in_p = iap_p->ep_in_p;
	result = config_ep_by_speed(
			cdev_p->gadget, &(iap_p->function), ep_in_p);
	if ((bool)unlikely(result != 0)) {
		DBG(cdev_p, "%s 1(%d)\n", __func__, result);
		goto fail_in;
	}

	result = usb_ep_enable(ep_in_p);

	if ((bool)unlikely(result != 0)) {
		DBG(cdev_p, "%s 2(%d)\n", __func__, result);
		goto fail_in;
	}

	ep_in_p->driver_data = iap_p;

	result = allocate_write_buffer(iap_p);
	if ((bool)unlikely(result != 0))
		goto fail_in_init;

	/* one bulk endpoint reads anything OUT (from the host) */
	ep_out_p = iap_p->ep_out_p;
	result = config_ep_by_speed(
			cdev_p->gadget, &(iap_p->function), ep_out_p);
	if ((bool)unlikely(result != 0)) {
		DBG(cdev_p, "%s 3(%d)\n", __func__, result);
		goto fail_out;
	}

	result = usb_ep_enable(ep_out_p);

	if ((bool)unlikely(result != 0)) {
		DBG(cdev_p, "%s 4(%d)\n", __func__, result);
		goto fail_out;
	}

	ep_out_p->driver_data = iap_p;

	result = allocate_read_buffer(iap_p);
	if ((bool)unlikely(result != 0))
		goto fail_out_init;

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	/* set connected */
	iap_p->is_connect = true;
	/* clear abort flag */
	iap_p->is_aborted = false;
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	set_connect_event(iap_p, APNUSB_EV_CONNECT);

	INFO(cdev_p, "%s enabled\n", iap_p->function.name);
	return result;

fail_out_init:
	if ((bool)unlikely(ep_out_p == NULL))
		goto fail_out;

	for (index_local = 0; index_local < ALPINE_IAP_MAX_READ_BUFF; index_local++) {
		if (iap_p->req_out_a[index_local] != NULL)
			usb_ep_dequeue(ep_out_p, iap_p->req_out_a[index_local]);
	}
	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	INIT_LIST_HEAD(&iap_p->read_buff_queue);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	usb_ep_disable(ep_out_p);

	for (index_local = 0; index_local < ALPINE_IAP_MAX_READ_BUFF; index_local++) {
		if (iap_p->req_out_a[index_local] != NULL)
			free_ep_req(ep_out_p, iap_p->req_out_a[index_local]);
	}

	ep_out_p->driver_data = NULL;

fail_out:

fail_in_init:
	if ((bool)unlikely(ep_in_p == NULL))
		goto fail_in;

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	INIT_LIST_HEAD(&iap_p->write_buff_pool);
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	usb_ep_disable(ep_in_p);

	for (index_local = 0; index_local < ALPINE_IAP_MAX_WRITE_BUFF; index_local++) {
		if (iap_p->req_in_a[index_local] != NULL)
			free_ep_req(ep_in_p, iap_p->req_in_a[index_local]);
	}

	ep_in_p->driver_data = NULL;

fail_in:
	DBG(cdev_p, "%s enable fail %d\n", iap_p->function.name, result);
	return result;
}

static void disable_alpine_iap(struct f_alpine_iap *iap_p)
{
	int				result;
	int				index_local;
	struct usb_composite_dev	*cdev_p;
	struct usb_ep			*ep_in_p;
	struct usb_ep			*ep_out_p;
	unsigned long			flags_local;

	cdev_p		= iap_p->function.config->cdev;
	ep_in_p		= iap_p->ep_in_p;
	ep_out_p	= iap_p->ep_out_p;

	/* set unconnected */
	iap_p->is_connect = false;
	/* clear abort flag */
	iap_p->is_aborted = false;

	/* Ep IN */
	if ((likely(ep_in_p != NULL)) &&
		(likely(ep_in_p->driver_data != NULL))) {
		/* request buffer */
		for (index_local = 0; index_local < ALPINE_IAP_MAX_WRITE_BUFF; index_local++)
			usb_ep_dequeue(ep_in_p, iap_p->req_in_a[index_local]);

		spin_lock_irqsave(&iap_p->spinlock, flags_local);
		INIT_LIST_HEAD(&iap_p->write_buff_pool);
		spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

		/* disable endpoint */
		result = usb_ep_disable(ep_in_p);
		if ((bool)unlikely(result != 0))
			DBG(cdev_p, "disable IN %s : %d\n",
					ep_in_p->name, result);

		for (index_local = 0; index_local < ALPINE_IAP_MAX_WRITE_BUFF; index_local++)
			free_ep_req(ep_in_p, iap_p->req_in_a[index_local]);

		ep_in_p->driver_data = NULL;
	}

	/* Ep OUT */
	if ((likely(ep_out_p != NULL)) &&
		(likely(ep_out_p->driver_data != NULL))) {
		/* request buffer */
		for (index_local = 0; index_local < ALPINE_IAP_MAX_READ_BUFF; index_local++)
			usb_ep_dequeue(ep_out_p, iap_p->req_out_a[index_local]);

		spin_lock_irqsave(&iap_p->spinlock, flags_local);
		INIT_LIST_HEAD(&iap_p->read_buff_queue);
		spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

		/* disable endpoint */
		result = usb_ep_disable(ep_out_p);
		if ((bool)unlikely(result != 0))
			DBG(cdev_p, "disable OUT %s : %d\n",
					ep_out_p->name, result);

		for (index_local = 0; index_local < ALPINE_IAP_MAX_READ_BUFF; index_local++)
			free_ep_req(ep_out_p, iap_p->req_out_a[index_local]);

		ep_out_p->driver_data = NULL;
	}

	spin_lock_irqsave(&iap_p->spinlock, flags_local);

	memset(iap_p->req_in_a,  0x00,
		(sizeof(struct usb_request *) * ALPINE_IAP_MAX_WRITE_BUFF));
	memset(iap_p->req_out_a, 0x00,
		(sizeof(struct usb_request *) * ALPINE_IAP_MAX_READ_BUFF));

	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	set_connect_event(iap_p, APNUSB_EV_DISCONNECT);

	wake_up(&iap_p->read_queue);
	wake_up(&iap_p->write_queue);

	INFO(cdev_p, "%s disabled\n", iap_p->function.name);
}

/*-------------------------------------------------------------------------*/

static void apn_device_release(struct device *dev_p)
{
	struct apn_device *apn_dev_p;

	dev_dbg(dev_p, "%s\n", __func__);

	apn_dev_p = container_of(dev_p, struct apn_device, dev);
	kfree(apn_dev_p);
}

static int apn_device_register(struct f_alpine_iap *iap_p)
{
	struct apn_device *apn_dev_p;
	int ret;
	unsigned long flags_local;

	apn_dev_p = kzalloc(sizeof(*apn_dev_p), GFP_KERNEL);
	if ((bool)unlikely(apn_dev_p == NULL)) {
		ret = -ENOMEM;
		goto fail_alloc;
	}

	device_initialize(&apn_dev_p->dev);
	apn_dev_p->dev.devt = MKDEV(apn_iap_major, apn_iap_minor);
	apn_dev_p->dev.class = apn_iap_class_p;
	apn_dev_p->dev.parent = NULL;
	apn_dev_p->dev.groups = NULL;
	apn_dev_p->dev.release = apn_device_release;
	dev_set_drvdata(&apn_dev_p->dev, iap_p);
	dev_set_name(&apn_dev_p->dev, APN_USB_DEVICE_NAME);

	cdev_init(&apn_dev_p->cdev, &f_alpine_iap_fops);

	ret = cdev_device_add(&apn_dev_p->cdev, &apn_dev_p->dev);
	if ((bool)unlikely(ret != 0)) {
		dev_err(&apn_dev_p->dev, "unable to add device\n");
		goto fail_add;
	}

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	iap_p->apn_dev_p = apn_dev_p;
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	return 0;

fail_add:
	put_device(&apn_dev_p->dev);
fail_alloc:
	return ret;
}

static void apn_device_unregister(struct f_alpine_iap *iap_p)
{
	struct apn_device *apn_dev_p = iap_p->apn_dev_p;
	unsigned long flags_local;

	cdev_device_del(&apn_dev_p->cdev, &apn_dev_p->dev);

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	iap_p->apn_dev_p = NULL;
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	put_device(&apn_dev_p->dev);
}

/*
 * We use this function as an alternative of usb_ep_autoconfig_release()
 * that is unavailable for out-of-tree kernel modules in case of GKI.
 */
static void apn_usb_ep_autoconfig_release(struct usb_ep *ep)
{
	ep->claimed = false;
	ep->driver_data = NULL;
}

/*-------------------------------------------------------------------------*/

static int alpine_iap_bind(struct usb_configuration *config_p,
			   struct usb_function *func_p)
{
	struct usb_composite_dev	*cdev_p;
	struct f_alpine_iap		*iap_p;
	struct usb_string		*us_p;
	struct usb_ep			*ep_in_p;
	struct usb_ep			*ep_out_p;
	int				id_local;
	int				ret;
	unsigned long			flags_local;

	ret		= 0;
	cdev_p		= config_p->cdev;
	iap_p		= func_to_alpiap(func_p);
	ep_in_p		= NULL;
	ep_out_p	= NULL;

	us_p = usb_gstrings_attach(cdev_p, alpine_iap_strings,
				 ARRAY_SIZE(strings_alpine_iap_defs));
	if (IS_ERR(us_p)) {
		ret =  PTR_ERR(us_p);
		goto err_id;
	}

	alpine_iap_interface_desc.iInterface = us_p[STRING_INTERFACE_IDX].id;

	/* allocate interface ID(s) */
	id_local = usb_interface_id(config_p, func_p);
	if ((bool)unlikely(id_local < 0)) {
		ret = id_local;
		goto err_id;
	}

	alpine_iap_interface_desc.bInterfaceNumber = id_local;

	/* allocate bulk endpoints */
	ep_in_p
		= usb_ep_autoconfig(cdev_p->gadget, &alpine_iap_ep_fs_in_desc);
	ep_out_p
		= usb_ep_autoconfig(cdev_p->gadget, &alpine_iap_ep_fs_out_desc);
	if ((bool)unlikely((ep_in_p == NULL) || (ep_out_p == NULL))) {
		ret = -ENODEV;
		goto fail;
	}

	/* support high speed hardware */
	alpine_iap_ep_hs_in_desc.bEndpointAddress
		= alpine_iap_ep_fs_in_desc.bEndpointAddress;
	alpine_iap_ep_hs_out_desc.bEndpointAddress
		= alpine_iap_ep_fs_out_desc.bEndpointAddress;

	/* assign */
	ret = usb_assign_descriptors(func_p, alpine_iap_fs_descs,
					alpine_iap_hs_descs, NULL, NULL);
	if ((bool)unlikely(ret != 0))
		goto fail;

	spin_lock_irqsave(&iap_p->spinlock, flags_local);

	iap_p->ep_in_p	= ep_in_p;
	iap_p->ep_out_p	= ep_out_p;
	iap_p->ep_in_p->driver_data = cdev_p;
	iap_p->ep_out_p->driver_data = cdev_p;

	memset(iap_p->req_in_a,  0x00,
		(sizeof(struct usb_request *) * ALPINE_IAP_MAX_WRITE_BUFF));
	memset(iap_p->req_out_a, 0x00,
		(sizeof(struct usb_request *) * ALPINE_IAP_MAX_READ_BUFF));

	INIT_LIST_HEAD(&iap_p->read_buff_queue);
	INIT_LIST_HEAD(&iap_p->write_buff_pool);

	DBG(cdev_p, "alpine iap %s speed: IN %s OUT %s\n",
		(gadget_is_superspeed(cdev_p->gadget) ? "super" :
		(gadget_is_dualspeed(cdev_p->gadget) ? "dual" : "full")),
		iap_p->ep_in_p->name, iap_p->ep_out_p->name);

	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	ret = apn_device_register(iap_p);
	if ((bool)unlikely(ret != 0))
		goto fail;

	return 0;

fail:
	if ((bool)likely(ep_out_p != NULL))
		apn_usb_ep_autoconfig_release(ep_out_p);

	if ((bool)likely(ep_in_p != NULL))
		apn_usb_ep_autoconfig_release(ep_in_p);
err_id:
	ERROR(cdev_p, "alpine iap can't bind. err=%d\n", ret);

	return ret;
}

static void alpine_iap_unbind(struct usb_configuration *config_p,
			      struct usb_function *func_p)
{
	int				index_local;
	struct apn_event		*event_p;
	struct apn_event		*next_p;
	struct f_alpine_iap		*iap_p;
	unsigned long			flags_local, flags_apnfd;

	iap_p = func_to_alpiap(func_p);

	spin_lock_irqsave(&iap_p->spinlock, flags_local);
	for (index_local = 0; index_local < ALPINE_IAP_OPEN_MAX; index_local++) {
		if (iap_p->apnfd_p[index_local] != NULL) {
			/* purge event list */
			spin_lock_irqsave(
				&iap_p->apnfd_p[index_local]->spinlock, flags_apnfd);
			list_for_each_entry_safe(event_p, next_p,
				&iap_p->apnfd_p[index_local]->event_list, list) {
				list_del(&event_p->list);
				kfree(event_p);
			}
			iap_p->apnfd_p[index_local]->fd_p->private_data = NULL;
			spin_unlock_irqrestore(
				&iap_p->apnfd_p[index_local]->spinlock, flags_apnfd);
			kfree(iap_p->apnfd_p[index_local]);
			iap_p->apnfd_p[index_local] = NULL;
			put_device(&iap_p->apn_dev_p->dev);
			iap_p->open_num--;
		}
	}
	spin_unlock_irqrestore(&iap_p->spinlock, flags_local);

	apn_device_unregister(iap_p);

	usb_ep_disable(iap_p->ep_in_p);

	usb_free_all_descriptors(func_p);

	DBG(config_p->cdev, "alpine iap unbind.\n");
}

static int alpine_iap_set_alt(struct usb_function *func_p,
			      unsigned intf, unsigned alt)
{
	struct f_alpine_iap		*iap_p;
	struct usb_composite_dev	*cdev_p;

	iap_p	= func_to_alpiap(func_p);
	cdev_p	= func_p->config->cdev;

	if ((bool)likely(iap_p->ep_in_p->driver_data != NULL))
		disable_alpine_iap(iap_p);

	DBG(cdev_p, "alpine iap set alternative %d.\n", alt);
	return enable_alpine_iap(cdev_p, iap_p);
}

static void alpine_iap_disable(struct usb_function *func_p)
{
	struct f_alpine_iap		*iap_p;

	iap_p	= func_to_alpiap(func_p);

	disable_alpine_iap(iap_p);

	DBG(func_p->config->cdev, "alpine iap disable\n");
}

static int alpine_iap_setup(struct usb_function *func_p,
			    const struct usb_ctrlrequest *ctrlreq_p)
{
	int				 result;
	struct usb_composite_dev	*cdev_p;

	result		= 0;
	cdev_p		= func_p->config->cdev;

	DBG(cdev_p, "%s ctrl_request : 0x%02x %02x 0x%04x\n",
			__func__,
			ctrlreq_p->bRequestType,
			ctrlreq_p->bRequest,
			ctrlreq_p->wValue);

	result = -EOPNOTSUPP;

	return result;
}

static void alpine_iap_suspend(struct usb_function *func_p)
{
	struct usb_composite_dev	*cdev_p;

	cdev_p = func_p->config->cdev;

	DBG(cdev_p, "%s call\n", __func__);
}

static void alpine_iap_resume(struct usb_function *func_p)
{
	struct usb_composite_dev	*cdev_p;

	cdev_p = func_p->config->cdev;

	DBG(cdev_p, "%s call\n", __func__);
}

static void alpine_iap_free(struct usb_function *f)
{
	struct f_alpine_iap *iap_p;
	struct f_alpine_opts *opts_p;

	iap_p = func_to_alpiap(f);
	opts_p = container_of(f->fi, struct f_alpine_opts, func_inst);

	iap_p->apn_dev_p = NULL;
	kfree(iap_p);

	mutex_lock(&opts_p->lock);
	opts_p->refcnt--;
	mutex_unlock(&opts_p->lock);
}

/*-------------------------------------------------------------------------*/

static struct usb_function *alpine_alloc(struct usb_function_instance *fi)
{
	struct f_alpine_iap		*iap_p;
	struct f_alpine_opts		*opts_p;
	int				index_local;

	/* allocate and initialize one new instance */
	iap_p = kzalloc(sizeof(struct f_alpine_iap), GFP_KERNEL);
	if ((bool)unlikely(iap_p == NULL))
		goto fail;

	opts_p = container_of(fi, struct f_alpine_opts, func_inst);
	mutex_lock(&opts_p->lock);
	opts_p->refcnt++;
	mutex_unlock(&opts_p->lock);

	/* initialize */
	iap_p->open_num			= 0;
	iap_p->is_connect		= false;
	iap_p->is_aborted		= false;
	for (index_local = 0; index_local < ALPINE_IAP_OPEN_MAX; index_local++)
		iap_p->apnfd_p[index_local] = NULL;
	iap_p->function.name		= ALPINE_IAP_NAME;
	iap_p->function.strings		= alpine_iap_strings;
	iap_p->function.bind		= alpine_iap_bind;
	iap_p->function.unbind		= alpine_iap_unbind;
	iap_p->function.set_alt		= alpine_iap_set_alt;
	iap_p->function.setup		= alpine_iap_setup;
	iap_p->function.disable		= alpine_iap_disable;
	iap_p->function.suspend		= alpine_iap_suspend;
	iap_p->function.resume		= alpine_iap_resume;
	iap_p->function.free_func	= alpine_iap_free;

	spin_lock_init(&iap_p->spinlock);
	init_waitqueue_head(&iap_p->read_queue);
	init_waitqueue_head(&iap_p->write_queue);

	return &iap_p->function;
fail:
	return ERR_PTR(-ENOMEM);
}
/*-------------------------------------------------------------------------*/

static int falpine_setup(void)
{
	int				 result;
	dev_t				 dev;

	/* Create Alpine iAP Class */
	apn_iap_class_p	= class_create(THIS_MODULE, APN_USB_DEVICE_NAME);
	if (IS_ERR(apn_iap_class_p)) {
		result = PTR_ERR(apn_iap_class_p);
		goto fail;
	}

	/* Allocate Alpine iAP Device */
	result = alloc_chrdev_region(&dev, 0, ALPINE_IAP_DEVICE_NUM,
						APN_USB_DEVICE_NAME);
	if (result != 0) {
		class_destroy(apn_iap_class_p);
		goto fail;
	} else {
		apn_iap_major	= MAJOR(dev);
		apn_iap_minor	= MINOR(dev);
	}

	return result;
fail:
	apn_iap_class_p = NULL;
	return result;
}

static void falpine_cleanup(void)
{
	/* unregist Alpine iAP Device */
	if ((bool)likely(apn_iap_major != 0)) {
		unregister_chrdev_region(MKDEV(apn_iap_major, apn_iap_minor),
						ALPINE_IAP_DEVICE_NUM);
		apn_iap_major = 0;
		apn_iap_minor = 0;
	} else {
		/* nothing to do */
	}

	/* Destroy Alpine iAP Class */
	class_destroy(apn_iap_class_p);
	apn_iap_class_p	= NULL;
}

/*-------------------------------------------------------------------------*/

static void alpine_free_inst(struct usb_function_instance *f)
{
	struct f_alpine_opts	*opts;

	opts = container_of(f, struct f_alpine_opts, func_inst);

	falpine_cleanup();

	kfree(opts);
}

static struct usb_function_instance *alpine_alloc_inst(void)
{
	struct f_alpine_opts		*opts;
	struct usb_function_instance	*ret;
	int status_local = 0;

	opts = kzalloc(sizeof(*opts), GFP_KERNEL);
	if (!opts) {
		ret = ERR_PTR(-ENOMEM);
		goto fail;
	}
	mutex_init(&opts->lock);
	opts->func_inst.free_func_inst = alpine_free_inst;

	status_local = falpine_setup();
	if ((bool)unlikely(status_local < 0)) {
		ret = ERR_PTR(status_local);
		goto err_setup;
	}

	config_group_init_type_name(&opts->func_inst.group,
					"", &alpine_func_type);

	return &opts->func_inst;
err_setup:
	kfree(opts);
fail:
	return ret;
}

DECLARE_USB_FUNCTION_INIT(iap, alpine_alloc_inst, alpine_alloc);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Akira Kanno");
MODULE_AUTHOR("Yuta Shimoyamada");
MODULE_AUTHOR("Tatsuji Kikuchi");
MODULE_AUTHOR("Yoshinori Iihoshi");
/* end of file */
